package com.zl.sdk.util;

import android.content.Context;
import android.os.Build;
import android.util.DisplayMetrics;
import android.util.Log;
import android.util.TypedValue;
import android.view.WindowManager;


/**
 * DpiUtils
 */
public class DpiUtil {

    private static final String TAG = "DpiUtils";

    private static Context mContext;
    // 当前屏幕的densityDpi
    private static float mDmDensityDpi = 0.0f;
    private static int mWidth = 0;
    private static int mHeight = 0;

    private static DisplayMetrics mDisplayMetrics;
    private static float mScale = 1.0f;

    // 设计分辨率
    private static int mDesignWidth = 0;
    private static int mDesignHeight = 0;
    private static float mScaleX = 1.0f;
    private static float mScaleY = 1.0f;

    public static boolean isFull() {
        float s = (float) getHeight() / getWidth();
        LogUtil.d("屏幕比列：" + s);
        LogUtil.d("屏幕height：" + DpiUtil.pxTodip(getHeight()));
        LogUtil.d("屏幕width：" + DpiUtil.pxTodip(getWidth()));
        return s > 2;
    }

    /**
     * 根据构造函数获得当前手机的屏幕系数
     */
    public static void init(Context context) {
        mContext = context;
        // 获取当前屏幕
        mDisplayMetrics = context.getResources().getDisplayMetrics();
        // 设置DensityDpi,mWidth,mHeight
        mDmDensityDpi = mDisplayMetrics.densityDpi;
        mWidth = mDisplayMetrics.widthPixels;
        mHeight = mDisplayMetrics.heightPixels;
        // 密度因子
        mScale = getDmDensityDpi() / 160;
        Log.i(TAG, " dmDensityDpi:" + mDmDensityDpi + " mWidth:" + mWidth
                + " mHeight:" + mHeight);
    }

    /**
     * 密度转换像素
     */
    public static int dipTopx(float dipValue) {
        return (int) (dipValue * mScale + 0.5f);
    }

    public static int sp2px(float sp) {
        return (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_SP, sp, mContext.getResources().getDisplayMetrics());
    }

    /**
     * 像素转换密度
     */
    public static int pxTodip(float pxValue) {
        return (int) (pxValue / mScale + 0.5f);
    }


    /**
     * 当前屏幕的density因子
     *
     * @retrun DmDensity Getter
     */
    public static float getDmDensityDpi() {
        return mDmDensityDpi;
    }

    /**
     * 获取设备的屏幕宽度(Dp)
     *
     * @return mWidth
     */
    public static int getDPWidth() {
        return pxTodip(mWidth);
    }


    /**
     * 获取设备的屏幕宽度(Dp)
     *
     * @return mWidth
     */
    public static int getDPHeight() {
        return pxTodip(mHeight);
    }

    /**
     * 获取设备的屏幕宽度(Dp)
     *
     * @return mWidth
     */
    public static int getScreenDPWidth() {
        return pxTodip(getScreenWidth());
    }


    /**
     * 获取设备的屏幕宽度(Dp)
     *
     * @return mWidth
     */
    public static int getScreenDPHeight() {
        return pxTodip(getScreenHeight());
    }


    /**
     * 获取屏幕宽度
     *
     * @return
     */
    public static int getScreenWidth() {
        int width;
        DisplayMetrics dm = new DisplayMetrics();
        WindowManager windowMgr = (WindowManager) mContext.getSystemService(Context.WINDOW_SERVICE);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            windowMgr.getDefaultDisplay().getRealMetrics(dm);
            width = dm.widthPixels;
        } else {
            width = DpiUtil.getWidth();
        }
        return width;
    }

    /**
     * 获取屏幕高度
     *
     * @return
     */
    public static int getScreenHeight() {
        int height;
        DisplayMetrics dm = new DisplayMetrics();
        WindowManager windowMgr = (WindowManager) mContext.getSystemService(Context.WINDOW_SERVICE);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            assert windowMgr != null;
            windowMgr.getDefaultDisplay().getRealMetrics(dm);
            // 获取高度
            height = dm.heightPixels;
        } else {
            height = DpiUtil.getHeight();
        }
        return height;
    }

    /**
     * 获取设备的屏幕宽度(Pixels)
     *
     * @return mWidth
     */
    public static int getWidth() {
        return mWidth;
    }


    /**
     * 获取设备的屏幕高度(Pixels)
     *
     * @return mHeight
     */
    public static int getHeight() {
        return mHeight;
    }

    /**
     * 设计分辨率
     */
    public static void SetDesignResolution(int width, int height) {
        mDesignWidth = width;
        mDesignHeight = height;
        mScaleX = (float) mWidth / mDesignWidth;
        mScaleY = (float) mHeight / mDesignHeight;
        Log.i(TAG, "mScaleX" + mScaleX + ", mScaleY" + mScaleY);
    }

    /**
     * 返回横向缩放比例
     *
     * @return
     */
    public static float getScaleX() {
        return mScaleX;
    }

    /**
     * 返回纵向缩放比例
     *
     * @return
     */
    public static float getScaleY() {
        return mScaleY;
    }

    /**
     * 将设计坐标点的横坐标x转换成实际像素点的x
     *
     * @param x
     * @return
     */
    public static int toRealX(int x) {
        return (int) (x * mScaleX);
    }

    /**
     * 将设计坐标点的纵坐标y转换成实际像素点的y
     *
     * @param y
     * @return
     */
    public static int toRealY(int y) {
        return (int) (y * mScaleY);
    }

    /**
     * 将实际像素点的x转换成设计坐标点的横坐标x
     *
     * @param x
     * @return
     */
    public static int toDesignX(int x) {
        return (int) (x / mScaleX);
    }

    /**
     * 将实际像素点的y转换成设计坐标点的纵坐标y
     *
     * @param y
     * @return
     */
    public static int toDesignY(int y) {
        return (int) (y / mScaleY);
    }

}
