package com.gem.media.splash.base.protocol;

import android.content.Context;
import android.graphics.Canvas;
import android.graphics.Color;
import android.view.MotionEvent;
import android.view.View;
import android.view.ViewGroup;
import android.widget.FrameLayout;

import com.gem.media.splash.base.utils.ALog;


public class ContentView extends FrameLayout {
    private static final String TAG = "ContentView";

    private boolean hasMask;
    private AdConfig config;
    private ViewGroup adContainer;

    public ContentView(Context context, AdConfig config) {
        super(context);
        this.hasMask = false;
        this.config = config;
        addAdContainer();
    }

    private void addAdContainer() {
        adContainer = new AdContainer(getContext());
        addView(adContainer, LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT);
    }

    private void addMask() {
        if (!hasMask) {
            addPermanentMask();
            addTemporaryMask();
            hasMask = true;
        }
    }

    private void addPermanentMask() {
        if (config.topHeightRate > 0 || config.bottomHeightRate > 0) {
            PermanentGroup permanentGroup = new PermanentGroup(getContext());
            addView(permanentGroup, LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT);

            if (config.topHeightRate > 0) {
                permanentGroup.addView(new PermanentMask(getContext(), PermanentMask.INDEX_LEFT));

                if (config.widthRate > 0 && config.highRate > 0) {
                    if (config.topHeightRate > config.highRate) {
                        permanentGroup.addView(new PermanentMask(getContext(), PermanentMask.INDEX_RIGHT));
                    }
                } else {
                    config.widthRate = 0;
                    config.highRate = 0;
                }
            }
            if (config.bottomHeightRate > 0) {
                permanentGroup.addView(new PermanentMask(getContext(), PermanentMask.INDEX_BOTTOM));
            }
        }
    }

    public void addTemporaryMask() {
        if (config.notClickInterval > 0 && Math.random() * 100 < config.notClickRate) {
            TemporaryMask mask = new TemporaryMask(getContext(), config.notClickInterval);
            addView(mask, LayoutParams.MATCH_PARENT, LayoutParams.MATCH_PARENT);
        }
    }

    public ViewGroup getAdContainer() {
        return adContainer;
    }

    public class AdContainer extends FrameLayout {
        public AdContainer(Context context) {
            super(context);
        }

        @Override
        public void addView(View child, int index, ViewGroup.LayoutParams params) {
            addMask();
            super.addView(child, index, params);
        }
    }

    public class PermanentGroup extends ViewGroup {
        public PermanentGroup(Context context) {
            super(context);
        }

        @Override
        protected void onMeasure(int widthMeasureSpec, int heightMeasureSpec) {
            int width = MeasureSpec.getSize(widthMeasureSpec);
            int height = MeasureSpec.getSize(heightMeasureSpec);
            for (int i = 0; i < getChildCount(); i++) {
                PermanentMask mask = (PermanentMask) getChildAt(i);
                int maskWidth;
                int maskHeight;
                switch (mask.index) {
                    case PermanentMask.INDEX_LEFT:
                        maskWidth = width - width * config.widthRate / 100;
                        maskHeight = height * config.topHeightRate / 100;
                        break;
                    case PermanentMask.INDEX_RIGHT:
                        maskWidth = width * config.widthRate / 100;
                        maskHeight = height * (config.topHeightRate - config.highRate) / 100;
                        break;
                    case PermanentMask.INDEX_BOTTOM:
                        maskWidth = width;
                        maskHeight = height * config.bottomHeightRate / 100;
                        break;
                    default:
                        continue;
                }
                measureChild(mask, MeasureSpec.makeMeasureSpec(maskWidth, MeasureSpec.EXACTLY), MeasureSpec.makeMeasureSpec(maskHeight, MeasureSpec.EXACTLY));
            }
            setMeasuredDimension(width, height);
        }

        @Override
        protected void onLayout(boolean changed, int left, int top, int right, int bottom) {
            int width = right - left;
            int height = bottom - top;
            for (int i = 0; i < getChildCount(); i++) {
                PermanentMask mask = (PermanentMask) getChildAt(i);
                int childLeft;
                int childTop;
                switch (mask.index) {
                    case PermanentMask.INDEX_LEFT:
                        childLeft = left;
                        childTop = top;
                        break;
                    case PermanentMask.INDEX_RIGHT:
                        childLeft = width - mask.getMeasuredWidth();
                        childTop = height * config.highRate / 100;
                        break;
                    case PermanentMask.INDEX_BOTTOM:
                        childLeft = left;
                        childTop = height - mask.getMeasuredHeight();
                        break;
                    default:
                        continue;
                }
                mask.layout(childLeft, childTop, childLeft + mask.getMeasuredWidth(), childTop + mask.getMeasuredHeight());
            }
        }
    }

    public class PermanentMask extends View {
        private static final int INDEX_LEFT = 0;
        private static final int INDEX_RIGHT = 1;
        private static final int INDEX_BOTTOM = 2;

        private int index;

        public PermanentMask(Context context, int index) {
            super(context);
            this.index = index;
                setBackgroundColor(Color.argb(80, 255, 0, 0));
        }

        @Override
        public boolean onTouchEvent(MotionEvent event) {
            ALog.d(TAG, "PermanentMask onTouchEvent");
            return true;
        }
    }

    public class TemporaryMask extends View {
        private long drawTime;
        private long interval;

        public TemporaryMask(Context context, long interval) {
            super(context);
            this.drawTime = 0;
            this.interval = interval;
                setBackgroundColor(Color.argb(80, 255, 255, 0));
        }

        @Override
        public boolean onTouchEvent(MotionEvent event) {
            ALog.d(TAG, "TemporaryMask onTouchEvent");
            if (drawTime > 0 && Math.abs(System.currentTimeMillis() - drawTime) > interval) {
                return false;
            }
            return true;
        }

        @Override
        protected void onDraw(Canvas canvas) {
            if (drawTime == 0) {
                drawTime = System.currentTimeMillis();
            }
            super.onDraw(canvas);
        }
    }
}
