package com.ads.cal.picturetranslate;

import android.app.Activity;
import android.view.ViewGroup;

import androidx.annotation.NonNull;

import com.thinkup.core.api.AdError;
import com.thinkup.core.api.TUAdInfo;
import com.thinkup.splashad.api.TUSplashAd;
import com.thinkup.splashad.api.TUSplashAdExtraInfo;
import com.thinkup.splashad.api.TUSplashAdListener;

import java.util.Date;


public class TopOnAppOpenAdManager {
    //
    private static final String LOG_TAG = "TopOnAppOpenAdManager";
    private static final String AD_UNIT_ID = "b1f138leh1o4gp";

    private TUSplashAd appOpenAd = null;
    private boolean isLoadingAd = false;
    private boolean isShowingAd = false;

    private OnLoadAdCompleteListener onLoadAdCompleteListener;
    private OnShowAdCompleteListener onShowAdCompleteListener;

    public boolean isShowingAd() {
        return isShowingAd;
    }

    /**
     * Keep track of the time an app open ad is loaded to ensure you don't show an expired ad.
     */
    private long loadTime = 0;

    /**
     * Constructor.
     */
    public TopOnAppOpenAdManager() {
    }

    private static final class AppOpenAdManagerHolder {
        static final TopOnAppOpenAdManager appOpenAdManager = new TopOnAppOpenAdManager();
    }

    public static TopOnAppOpenAdManager getInstance() {
        return AppOpenAdManagerHolder.appOpenAdManager;
    }


    /**
     * Load an ad.
     */
    public void loadAd() {
        // Do not load ad if there is an unused ad or one is already loading.
        if (isLoadingAd || isAdAvailable()) {
            return;
        }

        isLoadingAd = true;


        appOpenAd = new TUSplashAd(PictureApplication.getApplication(), AD_UNIT_ID, null);
        appOpenAd.setAdListener(new TUSplashAdListener() {
            @Override
            public void onAdLoaded(boolean b) {
                LogUtils.d(LOG_TAG, "onAdLoaded.");
                isLoadingAd = false;
                loadTime = (new Date()).getTime();
                if (null != onLoadAdCompleteListener) {
                    onLoadAdCompleteListener.onLoadAdComplete();
                    onLoadAdCompleteListener = null;
                }
            }

            @Override
            public void onAdLoadTimeout() {
                isLoadingAd = false;
                if (null != onLoadAdCompleteListener) {
                    onLoadAdCompleteListener.onLoadAdFailed("time out");
                    onLoadAdCompleteListener = null;
                }
                LogUtils.d(LOG_TAG, "onAdFailedToLoad: time out");
            }

            @Override
            public void onNoAdError(AdError adError) {
                isLoadingAd = false;
                if (null != onLoadAdCompleteListener) {
                    onLoadAdCompleteListener.onLoadAdFailed(adError.getPlatformMSG());
                    onLoadAdCompleteListener = null;
                }
                LogUtils.d(LOG_TAG, "onAdFailedToLoad: " + adError.getPlatformMSG());
            }

            @Override
            public void onAdShow(TUAdInfo atAdInfo) {
            }

            @Override
            public void onAdClick(TUAdInfo atAdInfo) {
            }

            @Override
            public void onAdDismiss(TUAdInfo atAdInfo, TUSplashAdExtraInfo atSplashAdExtraInfo) {
                appOpenAd = null;
                isShowingAd = false;

                LogUtils.d(LOG_TAG, "onAdDismissedFullScreenContent.");
                if (null != onShowAdCompleteListener) {
                    onShowAdCompleteListener.onShowAdComplete();
                    onShowAdCompleteListener = null;
                }
                loadAd();
            }
        });
        appOpenAd.loadAd();
    }

    /**
     * Check if ad was loaded more than n hours ago.
     */
    private boolean wasLoadTimeLessThanNHoursAgo(long numHours) {
        long dateDifference = (new Date()).getTime() - loadTime;
        long numMilliSecondsPerHour = 3600000;
        return (dateDifference < (numMilliSecondsPerHour * numHours));
    }

    /**
     * Check if ad exists and can be shown.
     */
    public boolean isAdAvailable() {
        return appOpenAd != null && wasLoadTimeLessThanNHoursAgo(4);
    }


    /**
     * Show the ad if one isn't already showing.
     *
     * @param activity the activity that shows the app open ad
     */
    public boolean showAdIfAvailable(@NonNull Activity activity) {
        if (isShowingAd) {
            LogUtils.d(LOG_TAG, "The app open ad is already showing.");
            return false;
        }

        if (!isAdAvailable()) {
            LogUtils.d(LOG_TAG, "The app open ad is not ready yet.");
            loadAd();
            return false;
        }

        LogUtils.d(LOG_TAG, "Will show ad.");
        isShowingAd = true;


        appOpenAd.show(activity, (ViewGroup) activity.getWindow().getDecorView());
        return true;
    }

    public interface OnShowAdCompleteListener {
        void onShowAdComplete();

        void onShowAdFailed(String errorMsg);
    }

    public interface OnLoadAdCompleteListener {
        void onLoadAdComplete();

        void onLoadAdFailed(String errorMsg);
    }

    public void setOnLoadAdCompleteListener(OnLoadAdCompleteListener onLoadAdCompleteListener) {
        this.onLoadAdCompleteListener = onLoadAdCompleteListener;
    }

    public void setOnShowAdCompleteListener(OnShowAdCompleteListener onShowAdCompleteListener) {
        this.onShowAdCompleteListener = onShowAdCompleteListener;
    }
}
