package com.ads.cal.picturetranslate;

import com.google.android.gms.tasks.OnFailureListener;
import com.google.android.gms.tasks.OnSuccessListener;
import com.google.android.gms.tasks.Task;
import com.google.mlkit.vision.common.InputImage;
import com.google.mlkit.vision.text.Text;
import com.google.mlkit.vision.text.TextRecognition;
import com.google.mlkit.vision.text.TextRecognizer;
import com.google.mlkit.vision.text.chinese.ChineseTextRecognizerOptions;
import com.google.mlkit.vision.text.devanagari.DevanagariTextRecognizerOptions;
import com.google.mlkit.vision.text.japanese.JapaneseTextRecognizerOptions;
import com.google.mlkit.vision.text.korean.KoreanTextRecognizerOptions;
import com.google.mlkit.vision.text.latin.TextRecognizerOptions;

public class PictureTranslateUtils {

    private final TextRecognizer recognizer;
    private final TextRecognizer chineseRecognizer;
    private final TextRecognizer devanagariRecognizer;
    private final TextRecognizer japaneseRecognizer;
    private final TextRecognizer koreanRecognizer;


    public PictureTranslateUtils() {
        recognizer = TextRecognition.getClient(TextRecognizerOptions.DEFAULT_OPTIONS);
        chineseRecognizer = TextRecognition.getClient(new ChineseTextRecognizerOptions.Builder().build());
        devanagariRecognizer = TextRecognition.getClient(new DevanagariTextRecognizerOptions.Builder().build());
        japaneseRecognizer = TextRecognition.getClient(new JapaneseTextRecognizerOptions.Builder().build());
        koreanRecognizer = TextRecognition.getClient(new KoreanTextRecognizerOptions.Builder().build());
    }

    private static final class PictureTranslateUtilsHolder {
        static final PictureTranslateUtils pictureTranslateUtils = new PictureTranslateUtils();
    }

    public static PictureTranslateUtils init() {
        return PictureTranslateUtilsHolder.pictureTranslateUtils;
    }

    public void translateStandard(InputImage image, OnSuccessListener<Text> onSuccessListener, OnFailureListener onFailureListener) {
        recognizer.process(image)
                .addOnSuccessListener(onSuccessListener)
                .addOnFailureListener(onFailureListener);
    }

    public void translateChinese(InputImage image, OnSuccessListener<Text> onSuccessListener, OnFailureListener onFailureListener) {
        chineseRecognizer.process(image)
                .addOnSuccessListener(onSuccessListener)
                .addOnFailureListener(onFailureListener);
    }

    public void translateDevanagari(InputImage image, OnSuccessListener<Text> onSuccessListener, OnFailureListener onFailureListener) {
        devanagariRecognizer.process(image)
                .addOnSuccessListener(onSuccessListener)
                .addOnFailureListener(onFailureListener);
    }

    public void translateJapanese(InputImage image, OnSuccessListener<Text> onSuccessListener, OnFailureListener onFailureListener) {
        japaneseRecognizer.process(image)
                .addOnSuccessListener(onSuccessListener)
                .addOnFailureListener(onFailureListener);
    }

    public void translateKorean(InputImage image, OnSuccessListener<Text> onSuccessListener, OnFailureListener onFailureListener) {
        koreanRecognizer.process(image)
                .addOnSuccessListener(onSuccessListener)
                .addOnFailureListener(onFailureListener);
    }

    public void close() {
        if (recognizer != null) {
            recognizer.close();
        }
        if (chineseRecognizer != null) {
            chineseRecognizer.close();
        }
        if (devanagariRecognizer != null) {
            devanagariRecognizer.close();
        }
        if (japaneseRecognizer != null) {
            japaneseRecognizer.close();
        }
        if (koreanRecognizer != null) {
            koreanRecognizer.close();
        }
    }


}
