package com.ads.cal.decibel.ui;

import android.app.Activity;
import android.util.Log;

import androidx.annotation.NonNull;

import com.google.android.gms.ads.AdError;
import com.google.android.gms.ads.AdRequest;
import com.google.android.gms.ads.FullScreenContentCallback;
import com.google.android.gms.ads.LoadAdError;
import com.google.android.gms.ads.appopen.AppOpenAd;

import java.util.Date;

public class OpenManager {
    private static final String LOG_TAG = "OpenManager";
//    private static final String AD_UNIT_ID = "ca-app-pub-3940256099942544/3419835294";

    /**
     * Constructor.
     */
    public OpenManager() {
    }


    private static final class OHolder {
        static final OpenManager o = new OpenManager();
    }

    public static OpenManager getInstance() {
        return OHolder.o;
    }

    /**
     * Keep track of the time an app open ad is loaded to ensure you don't show an expired ad.
     */
    private long loadTime = 0;


    private AppOpenAd appOpenAd = null;
    private boolean isLoadingAd = false;
    private boolean isShowingAd = false;

    private OnAdLoadCompleteCallBack onAdLoadCompleteCallBack;
    private OnAdShowCompleteCallBack onAdShowCompleteCallBack;

    public boolean isShowingAd() {
        return isShowingAd;
    }

    /**
     * Load an ad.
     *
     */
    public void loadAd() {
        // Do not load ad if there is an unused ad or one is already loading.
        if (isLoadingAd || isAdAvailable()) {
            return;
        }

        isLoadingAd = true;
        AdRequest request = new AdRequest.Builder().build();
        AppOpenAd.load(
                A.context,
                "",
                request,
                new AppOpenAd.AppOpenAdLoadCallback() {
                    /**
                     * Called when an app open ad has loaded.
                     *
                     * @param ad the loaded app open ad.
                     */
                    @Override
                    public void onAdLoaded(AppOpenAd ad) {
                        appOpenAd = ad;
                        isLoadingAd = false;
                        loadTime = (new Date()).getTime();
                        if (null != onAdLoadCompleteCallBack) {
                            onAdLoadCompleteCallBack.onLoadAdComplete();
                            onAdLoadCompleteCallBack = null;
                        }

                        Log.e(LOG_TAG, "onAdLoaded.");
                    }

                    /**
                     * Called when an app open ad has failed to load.
                     *
                     * @param loadAdError the error.
                     */
                    @Override
                    public void onAdFailedToLoad(LoadAdError loadAdError) {
                        isLoadingAd = false;
                        if (null != onAdLoadCompleteCallBack) {
                            onAdLoadCompleteCallBack.onLoadAdFailed(loadAdError.getMessage());
                            onAdLoadCompleteCallBack = null;
                        }
                        Log.e(LOG_TAG, "onAdFailedToLoad: " + loadAdError.getMessage());
                    }
                });
    }

    /**
     * Check if ad was loaded more than n hours ago.
     */
    private boolean wasLoadTimeLessThanNHoursAgo(long numHours) {
        long dateDifference = (new Date()).getTime() - loadTime;
        long numMilliSecondsPerHour = 3600000;
        return (dateDifference < (numMilliSecondsPerHour * numHours));
    }

    /**
     * Check if ad exists and can be shown.
     */
    public boolean isAdAvailable() {
        // Ad references in the app open beta will time out after four hours, but this time limit
        // may change in future beta versions. For details, see:
        // https://support.google.com/admob/answer/9341964?hl=en
        return appOpenAd != null && wasLoadTimeLessThanNHoursAgo(4);
    }



    /**
     * Show the ad if one isn't already showing.
     *
     * @param activity                 the activity that shows the app open ad
     */
    public boolean showAdIfAvailable(
            @NonNull Activity activity) {
        // If the app open ad is already showing, do not show the ad again.
        if (isShowingAd) {
            Log.e(LOG_TAG, "The app open ad is already showing.");
            return false;
        }

        // If the app open ad is not available yet, invoke the callback then load the ad.
        if (!isAdAvailable()) {
            Log.e(LOG_TAG, "The app open ad is not ready yet.");
            loadAd();
            return false;
        }

        Log.e(LOG_TAG, "Will show ad.");

        appOpenAd.setFullScreenContentCallback(
                new FullScreenContentCallback() {
                    /** Called when full screen content is dismissed. */
                    @Override
                    public void onAdDismissedFullScreenContent() {
                        // Set the reference to null so isAdAvailable() returns false.
                        appOpenAd = null;
                        isShowingAd = false;

                        Log.e(LOG_TAG, "onAdDismissedFullScreenContent.");
                        if (null != onAdShowCompleteCallBack) {
                            onAdShowCompleteCallBack.onShowAdComplete();
                            onAdShowCompleteCallBack = null;
                        }
                        loadAd();
                    }

                    /** Called when fullscreen content failed to show. */
                    @Override
                    public void onAdFailedToShowFullScreenContent(@NonNull AdError adError) {
                        appOpenAd = null;
                        isShowingAd = false;

                        Log.e(LOG_TAG, "onAdFailedToShowFullScreenContent: " + adError.getMessage());
                        if (null != onAdShowCompleteCallBack) {
                            onAdShowCompleteCallBack.onShowAdFailed(adError.getMessage());
                            onAdShowCompleteCallBack = null;
                        }
                        loadAd();
                    }

                    /** Called when fullscreen content is shown. */
                    @Override
                    public void onAdShowedFullScreenContent() {
                        Log.e(LOG_TAG, "onAdShowedFullScreenContent.");
                    }

                    @Override
                    public void onAdImpression() {
                        super.onAdImpression();
                        Log.e(LOG_TAG, "onAdImpression.");
                    }

                    @Override
                    public void onAdClicked() {
                        super.onAdClicked();
                    }
                });

        isShowingAd = true;
        appOpenAd.show(activity);
        return true;
    }

    public interface OnAdShowCompleteCallBack {
        void onShowAdComplete();

        void onShowAdFailed(String errorMsg);
    }

    public interface OnAdLoadCompleteCallBack {
        void onLoadAdComplete();

        void onLoadAdFailed(String errorMsg);
    }

    public void setOnLoadAdCompleteCallBack(OnAdLoadCompleteCallBack onAdLoadCompleteCallBack) {
        this.onAdLoadCompleteCallBack = onAdLoadCompleteCallBack;
    }

    public void setOnShowAdCompleteCallBack(OnAdShowCompleteCallBack onAdShowCompleteCallBack) {
        this.onAdShowCompleteCallBack = onAdShowCompleteCallBack;
    }
}
